/**
 * MXroute Alias Manager - Popup
 */

// DOM Elements
const setupRequired = document.getElementById('setup-required');
const mainContent = document.getElementById('main-content');
const openSettingsBtn = document.getElementById('open-settings');
const settingsBtn = document.getElementById('settings-btn');

const currentDomainSpan = document.getElementById('current-domain');
const aliasInput = document.getElementById('alias-input');
const selectedDomainSpan = document.getElementById('selected-domain');
const domainSelect = document.getElementById('domain-select');
const createBtn = document.getElementById('create-btn');
const autoGenerateBtn = document.getElementById('auto-generate-btn');
const createStatus = document.getElementById('create-status');

const recentList = document.getElementById('recent-list');
const viewAllBtn = document.getElementById('view-all-btn');

// State
let currentTabDomain = '';
let selectedDomain = '';
let domains = [];
let settings = {};

/**
 * Initialize the popup
 */
async function init() {
  // Set up event listeners
  openSettingsBtn.addEventListener('click', openSettings);
  settingsBtn.addEventListener('click', openSettings);
  createBtn.addEventListener('click', handleCreateAlias);
  autoGenerateBtn.addEventListener('click', handleAutoGenerate);
  domainSelect.addEventListener('change', handleDomainChange);
  viewAllBtn.addEventListener('click', () => chrome.runtime.openOptionsPage());

  aliasInput.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') {
      handleCreateAlias();
    }
  });

  // Check if configured
  const { configured } = await sendMessage({ action: 'isConfigured' });

  if (!configured) {
    setupRequired.classList.remove('hidden');
    mainContent.classList.add('hidden');
    return;
  }

  setupRequired.classList.add('hidden');
  mainContent.classList.remove('hidden');

  // Load initial data
  await Promise.all([
    loadCurrentTab(),
    loadDomains(),
    loadSettings()
  ]);

  await loadRecentAliases();

  // Auto-generate alias if format is not manual
  if (settings.aliasFormat !== 'manual' && currentTabDomain) {
    handleAutoGenerate();
  }
}

/**
 * Open settings page
 */
function openSettings() {
  chrome.runtime.openOptionsPage();
}

/**
 * Load current tab domain
 */
async function loadCurrentTab() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab && tab.url) {
      const url = new URL(tab.url);
      currentTabDomain = url.hostname.replace(/^www\./, '');
      currentDomainSpan.textContent = currentTabDomain;
    }
  } catch (error) {
    currentDomainSpan.textContent = '-';
  }
}

/**
 * Load available domains
 */
async function loadDomains() {
  try {
    domains = await sendMessage({ action: 'getDomains' });
    settings = await sendMessage({ action: 'getSettings' });
    const destinations = await sendMessage({ action: 'getDomainDestinations' });

    // Clear and populate domain select
    domainSelect.textContent = '';

    if (!domains || domains.length === 0) {
      const opt = document.createElement('option');
      opt.value = '';
      opt.textContent = 'No domains';
      domainSelect.appendChild(opt);
      return;
    }

    // Filter domains that have destinations configured
    const domainsWithDest = domains.filter(d => {
      const name = typeof d === 'string' ? d : d.domain;
      return destinations[name];
    });

    // Use domains with destinations, or all domains if none configured
    const availableDomains = domainsWithDest.length > 0 ? domainsWithDest : domains;

    for (const domain of availableDomains) {
      const domainName = typeof domain === 'string' ? domain : domain.domain;
      const opt = document.createElement('option');
      opt.value = domainName;
      opt.textContent = domainName;
      domainSelect.appendChild(opt);
    }

    // Set default domain
    if (settings.defaultDomain && availableDomains.some(d =>
      (typeof d === 'string' ? d : d.domain) === settings.defaultDomain
    )) {
      selectedDomain = settings.defaultDomain;
    } else {
      const firstDomain = availableDomains[0];
      selectedDomain = typeof firstDomain === 'string' ? firstDomain : firstDomain.domain;
    }

    domainSelect.value = selectedDomain;
    selectedDomainSpan.textContent = selectedDomain;
  } catch (error) {
    console.error('Error loading domains:', error);
    domainSelect.textContent = '';
    const opt = document.createElement('option');
    opt.value = '';
    opt.textContent = 'Error loading';
    domainSelect.appendChild(opt);
  }
}

/**
 * Load settings
 */
async function loadSettings() {
  settings = await sendMessage({ action: 'getSettings' });
}

/**
 * Handle domain selection change
 */
function handleDomainChange() {
  selectedDomain = domainSelect.value;
  selectedDomainSpan.textContent = selectedDomain;
}

/**
 * Handle auto-generate alias
 */
async function handleAutoGenerate() {
  if (!currentTabDomain) {
    return;
  }

  const alias = await sendMessage({
    action: 'generateAlias',
    site: currentTabDomain
  });

  aliasInput.value = alias || '';
  aliasInput.focus();
}

/**
 * Handle creating an alias
 */
async function handleCreateAlias() {
  const alias = aliasInput.value.trim();

  if (!alias) {
    showStatus('Please enter an alias', 'error');
    return;
  }

  if (!selectedDomain) {
    showStatus('Please select a domain', 'error');
    return;
  }

  // Get destination for this domain
  const destinations = await sendMessage({ action: 'getDomainDestinations' });
  const destination = destinations[selectedDomain];

  if (!destination) {
    showStatus('No destination set for this domain. Configure in settings.', 'error');
    return;
  }

  // Disable UI
  createBtn.disabled = true;
  aliasInput.disabled = true;

  try {
    const result = await sendMessage({
      action: 'createAlias',
      domain: selectedDomain,
      alias: alias,
      destination: destination,
      site: currentTabDomain
    });

    const fullAlias = result.fullAlias || `${alias}@${selectedDomain}`;

    // Copy to clipboard
    await copyToClipboard(fullAlias);

    // Show success with copy button
    showStatus(`Created: ${fullAlias}`, 'success', fullAlias);

    // Clear input
    aliasInput.value = '';

    // Reload recent aliases
    await loadRecentAliases();

    // Send to content script to fill field
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab) {
        chrome.tabs.sendMessage(tab.id, {
          action: 'fillEmail',
          email: fullAlias
        });
      }
    } catch {
      // Content script may not be available
    }
  } catch (error) {
    showStatus(`Error: ${error.message}`, 'error');
  } finally {
    createBtn.disabled = false;
    aliasInput.disabled = false;
    aliasInput.focus();
  }
}

/**
 * Load recent aliases
 */
async function loadRecentAliases() {
  try {
    const recent = await sendMessage({ action: 'getRecentAliases', limit: 5 });

    if (!recent || recent.length === 0) {
      recentList.textContent = '';
      const emptyDiv = document.createElement('div');
      emptyDiv.className = 'empty-recent';
      emptyDiv.textContent = 'No aliases yet';
      recentList.appendChild(emptyDiv);
      return;
    }

    recentList.textContent = '';
    for (const alias of recent) {
      recentList.appendChild(createRecentItem(alias));
    }
  } catch (error) {
    console.error('Error loading recent aliases:', error);
    recentList.textContent = '';
    const emptyDiv = document.createElement('div');
    emptyDiv.className = 'empty-recent';
    emptyDiv.textContent = 'Error loading';
    recentList.appendChild(emptyDiv);
  }
}

/**
 * Create a recent alias item element
 */
function createRecentItem(alias) {
  const email = `${alias.alias}@${alias.domain}`;

  const item = document.createElement('div');
  item.className = 'recent-item';

  const infoDiv = document.createElement('div');
  infoDiv.className = 'recent-item-info';

  const aliasSpan = document.createElement('span');
  aliasSpan.className = 'recent-alias';
  aliasSpan.textContent = alias.alias;
  aliasSpan.title = email;
  infoDiv.appendChild(aliasSpan);

  if (alias.site) {
    const siteSpan = document.createElement('span');
    siteSpan.className = 'recent-site';
    siteSpan.textContent = alias.site;
    infoDiv.appendChild(siteSpan);
  }

  item.appendChild(infoDiv);

  const actionsDiv = document.createElement('div');
  actionsDiv.className = 'recent-actions';

  const copyBtn = document.createElement('button');
  copyBtn.className = 'btn-icon';
  copyBtn.title = 'Copy to clipboard';
  copyBtn.textContent = '\u{1F4CB}';
  copyBtn.addEventListener('click', async (e) => {
    e.stopPropagation();
    await copyToClipboard(email);
    copyBtn.textContent = '\u2713';
    setTimeout(() => {
      copyBtn.textContent = '\u{1F4CB}';
    }, 1000);
  });
  actionsDiv.appendChild(copyBtn);

  const deleteBtn = document.createElement('button');
  deleteBtn.className = 'btn-icon delete';
  deleteBtn.title = 'Delete alias';
  deleteBtn.textContent = '\u{1F5D1}';
  deleteBtn.addEventListener('click', async (e) => {
    e.stopPropagation();
    if (confirm(`Delete ${email}?`)) {
      try {
        await sendMessage({
          action: 'deleteAlias',
          domain: alias.domain,
          alias: alias.alias
        });
        await loadRecentAliases();
      } catch (error) {
        showStatus(`Error: ${error.message}`, 'error');
      }
    }
  });
  actionsDiv.appendChild(deleteBtn);

  item.appendChild(actionsDiv);
  return item;
}

/**
 * Show status message
 */
function showStatus(message, type, copyValue = null) {
  createStatus.textContent = '';
  createStatus.className = `status ${type}`;
  createStatus.classList.remove('hidden');

  const textSpan = document.createElement('span');
  textSpan.textContent = message;
  createStatus.appendChild(textSpan);

  if (copyValue) {
    const copyBtn = document.createElement('button');
    copyBtn.className = 'copy-btn';
    copyBtn.textContent = '\u{1F4CB}';
    copyBtn.title = 'Copy';
    copyBtn.addEventListener('click', async () => {
      await copyToClipboard(copyValue);
      copyBtn.textContent = '\u2713';
    });
    createStatus.appendChild(copyBtn);
  }

  setTimeout(() => {
    createStatus.classList.add('hidden');
  }, 5000);
}

/**
 * Copy text to clipboard
 */
async function copyToClipboard(text) {
  try {
    await navigator.clipboard.writeText(text);
  } catch {
    // Fallback
    const textarea = document.createElement('textarea');
    textarea.value = text;
    document.body.appendChild(textarea);
    textarea.select();
    document.execCommand('copy');
    document.body.removeChild(textarea);
  }
}

/**
 * Send message to background script
 */
function sendMessage(message) {
  return new Promise((resolve, reject) => {
    chrome.runtime.sendMessage(message, response => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else if (response && response.error) {
        reject(new Error(response.error));
      } else {
        resolve(response);
      }
    });
  });
}

// Initialize
document.addEventListener('DOMContentLoaded', init);
