/**
 * MXroute Alias Manager - Options Page
 */

// DOM Elements
const tabs = document.querySelectorAll('.tab');
const tabContents = document.querySelectorAll('.tab-content');

// Credentials elements
const credentialsForm = document.getElementById('credentials-form');
const serverInput = document.getElementById('server');
const usernameInput = document.getElementById('username');
const apiKeyInput = document.getElementById('apiKey');
const testConnectionBtn = document.getElementById('test-connection');
const credentialsStatus = document.getElementById('credentials-status');

// Settings elements
const settingsForm = document.getElementById('settings-form');
const aliasFormatSelect = document.getElementById('aliasFormat');
const defaultDomainSelect = document.getElementById('defaultDomain');
const autoDetectCheckbox = document.getElementById('autoDetect');
const settingsStatus = document.getElementById('settings-status');

// Domains elements
const domainsList = document.getElementById('domains-list');
const saveDomainsBtn = document.getElementById('save-domains');
const domainsStatus = document.getElementById('domains-status');

// Aliases elements
const aliasSearch = document.getElementById('alias-search');
const aliasFilter = document.getElementById('alias-filter');
const aliasesList = document.getElementById('aliases-list');
const aliasesStatus = document.getElementById('aliases-status');

// Cache
let cachedDomains = [];
let cachedAliases = [];

/**
 * Initialize the options page
 */
async function init() {
  setupTabs();
  await loadCredentials();
  await loadSettings();

  // Set up event listeners
  credentialsForm.addEventListener('submit', handleSaveCredentials);
  testConnectionBtn.addEventListener('click', handleTestConnection);
  settingsForm.addEventListener('submit', handleSaveSettings);
  saveDomainsBtn.addEventListener('click', handleSaveDomains);
  aliasSearch.addEventListener('input', debounce(filterAliases, 300));
  aliasFilter.addEventListener('change', filterAliases);
}

/**
 * Set up tab navigation
 */
function setupTabs() {
  tabs.forEach(tab => {
    tab.addEventListener('click', () => {
      const targetTab = tab.dataset.tab;

      // Update active states
      tabs.forEach(t => t.classList.remove('active'));
      tabContents.forEach(tc => tc.classList.remove('active'));

      tab.classList.add('active');
      document.getElementById(targetTab).classList.add('active');

      // Load tab-specific content
      if (targetTab === 'domains') {
        loadDomains();
      } else if (targetTab === 'aliases') {
        loadAliases();
      } else if (targetTab === 'settings') {
        loadDomainsForSettings();
      }
    });
  });
}

/**
 * Load saved credentials
 */
async function loadCredentials() {
  const credentials = await sendMessage({ action: 'getCredentials' });
  if (credentials) {
    serverInput.value = credentials.server || '';
    usernameInput.value = credentials.username || '';
    apiKeyInput.value = credentials.apiKey || '';
  }
}

/**
 * Load saved settings
 */
async function loadSettings() {
  const settings = await sendMessage({ action: 'getSettings' });
  aliasFormatSelect.value = settings.aliasFormat || 'domain';
  autoDetectCheckbox.checked = settings.autoDetect !== false;

  // Load domains for default domain selector
  await loadDomainsForSettings();

  if (settings.defaultDomain) {
    defaultDomainSelect.value = settings.defaultDomain;
  }
}

/**
 * Load domains for settings page
 */
async function loadDomainsForSettings() {
  try {
    const { configured } = await sendMessage({ action: 'isConfigured' });
    if (!configured) {
      defaultDomainSelect.textContent = '';
      const opt = document.createElement('option');
      opt.value = '';
      opt.textContent = 'Configure credentials first';
      defaultDomainSelect.appendChild(opt);
      return;
    }

    const domains = await sendMessage({ action: 'getDomains' });
    cachedDomains = domains || [];

    defaultDomainSelect.textContent = '';
    const defaultOpt = document.createElement('option');
    defaultOpt.value = '';
    defaultOpt.textContent = 'Select default domain...';
    defaultDomainSelect.appendChild(defaultOpt);

    for (const domain of cachedDomains) {
      const domainName = typeof domain === 'string' ? domain : domain.domain;
      const option = document.createElement('option');
      option.value = domainName;
      option.textContent = domainName;
      defaultDomainSelect.appendChild(option);
    }

    // Set selected value
    const settings = await sendMessage({ action: 'getSettings' });
    if (settings.defaultDomain) {
      defaultDomainSelect.value = settings.defaultDomain;
    }
  } catch (error) {
    defaultDomainSelect.textContent = '';
    const opt = document.createElement('option');
    opt.value = '';
    opt.textContent = 'Error loading domains';
    defaultDomainSelect.appendChild(opt);
    console.error('Error loading domains:', error);
  }
}

/**
 * Handle saving credentials
 */
async function handleSaveCredentials(e) {
  e.preventDefault();

  const credentials = {
    server: serverInput.value.trim(),
    username: usernameInput.value.trim(),
    apiKey: apiKeyInput.value.trim()
  };

  if (!credentials.server || !credentials.username || !credentials.apiKey) {
    showStatus(credentialsStatus, 'Please fill in all fields', 'error');
    return;
  }

  try {
    await sendMessage({ action: 'saveCredentials', credentials });
    showStatus(credentialsStatus, 'Credentials saved successfully', 'success');

    // Reload domains
    await loadDomainsForSettings();
  } catch (error) {
    showStatus(credentialsStatus, `Error: ${error.message}`, 'error');
  }
}

/**
 * Handle testing connection
 */
async function handleTestConnection() {
  const credentials = {
    server: serverInput.value.trim(),
    username: usernameInput.value.trim(),
    apiKey: apiKeyInput.value.trim()
  };

  if (!credentials.server || !credentials.username || !credentials.apiKey) {
    showStatus(credentialsStatus, 'Please fill in all fields', 'error');
    return;
  }

  testConnectionBtn.disabled = true;
  testConnectionBtn.textContent = '';
  const spinner = document.createElement('span');
  spinner.className = 'loading';
  testConnectionBtn.appendChild(spinner);
  testConnectionBtn.appendChild(document.createTextNode(' Testing...'));

  try {
    const result = await sendMessage({ action: 'testConnection', credentials });
    if (result.success) {
      showStatus(credentialsStatus, 'Connection successful!', 'success');
    } else {
      const errorMsg = result.error || 'Check your credentials.';
      showStatus(credentialsStatus, `Connection failed: ${errorMsg}`, 'error');
    }
  } catch (error) {
    showStatus(credentialsStatus, `Error: ${error.message}`, 'error');
  } finally {
    testConnectionBtn.disabled = false;
    testConnectionBtn.textContent = 'Test Connection';
  }
}

/**
 * Handle saving settings
 */
async function handleSaveSettings(e) {
  e.preventDefault();

  const settings = {
    aliasFormat: aliasFormatSelect.value,
    defaultDomain: defaultDomainSelect.value || null,
    autoDetect: autoDetectCheckbox.checked
  };

  try {
    await sendMessage({ action: 'saveSettings', settings });
    showStatus(settingsStatus, 'Settings saved successfully', 'success');
  } catch (error) {
    showStatus(settingsStatus, `Error: ${error.message}`, 'error');
  }
}

/**
 * Create a DOM element from safe data
 */
function createDomainItem(domainName, destination) {
  const item = document.createElement('div');
  item.className = 'domain-item';

  const nameSpan = document.createElement('span');
  nameSpan.className = 'domain-name';
  nameSpan.textContent = domainName;
  item.appendChild(nameSpan);

  const destDiv = document.createElement('div');
  destDiv.className = 'domain-destination';

  const label = document.createElement('label');
  label.textContent = 'Forward to:';
  destDiv.appendChild(label);

  const input = document.createElement('input');
  input.type = 'email';
  input.className = 'input destination-input';
  input.dataset.domain = domainName;
  input.value = destination;
  input.placeholder = 'your-email@example.com';
  destDiv.appendChild(input);

  item.appendChild(destDiv);
  return item;
}

/**
 * Load domains with their destinations
 */
async function loadDomains() {
  domainsList.textContent = '';
  const loadingDiv = document.createElement('div');
  loadingDiv.className = 'loading-state';
  const spinner = document.createElement('div');
  spinner.className = 'loading';
  loadingDiv.appendChild(spinner);
  loadingDiv.appendChild(document.createTextNode(' Loading domains...'));
  domainsList.appendChild(loadingDiv);

  try {
    const { configured } = await sendMessage({ action: 'isConfigured' });
    if (!configured) {
      domainsList.textContent = '';
      const emptyDiv = document.createElement('div');
      emptyDiv.className = 'empty-state';
      const h3 = document.createElement('h3');
      h3.textContent = 'Not configured';
      const p = document.createElement('p');
      p.textContent = 'Please set up your API credentials first.';
      emptyDiv.appendChild(h3);
      emptyDiv.appendChild(p);
      domainsList.appendChild(emptyDiv);
      return;
    }

    const domains = await sendMessage({ action: 'getDomains' });
    const destinations = await sendMessage({ action: 'getDomainDestinations' });

    cachedDomains = domains || [];

    if (cachedDomains.length === 0) {
      domainsList.textContent = '';
      const emptyDiv = document.createElement('div');
      emptyDiv.className = 'empty-state';
      const h3 = document.createElement('h3');
      h3.textContent = 'No domains found';
      const p = document.createElement('p');
      p.textContent = 'No domains are associated with this account.';
      emptyDiv.appendChild(h3);
      emptyDiv.appendChild(p);
      domainsList.appendChild(emptyDiv);
      return;
    }

    domainsList.textContent = '';
    for (const domain of cachedDomains) {
      const domainName = typeof domain === 'string' ? domain : domain.domain;
      const destination = destinations[domainName] || '';
      domainsList.appendChild(createDomainItem(domainName, destination));
    }
  } catch (error) {
    domainsList.textContent = '';
    const emptyDiv = document.createElement('div');
    emptyDiv.className = 'empty-state';
    const h3 = document.createElement('h3');
    h3.textContent = 'Error';
    const p = document.createElement('p');
    p.textContent = error.message;
    emptyDiv.appendChild(h3);
    emptyDiv.appendChild(p);
    domainsList.appendChild(emptyDiv);
  }
}

/**
 * Handle saving domain destinations
 */
async function handleSaveDomains() {
  const inputs = domainsList.querySelectorAll('.destination-input');
  const destinations = {};

  inputs.forEach(input => {
    const domain = input.dataset.domain;
    const destination = input.value.trim();
    if (destination) {
      destinations[domain] = destination;
    }
  });

  try {
    await sendMessage({ action: 'saveDomainDestinations', destinations });
    showStatus(domainsStatus, 'Destinations saved successfully', 'success');
  } catch (error) {
    showStatus(domainsStatus, `Error: ${error.message}`, 'error');
  }
}

/**
 * Load aliases
 */
async function loadAliases() {
  aliasesList.textContent = '';
  const loadingDiv = document.createElement('div');
  loadingDiv.className = 'loading-state';
  const spinner = document.createElement('div');
  spinner.className = 'loading';
  loadingDiv.appendChild(spinner);
  loadingDiv.appendChild(document.createTextNode(' Loading aliases...'));
  aliasesList.appendChild(loadingDiv);

  try {
    const { configured } = await sendMessage({ action: 'isConfigured' });
    if (!configured) {
      aliasesList.textContent = '';
      const emptyDiv = document.createElement('div');
      emptyDiv.className = 'empty-state';
      const h3 = document.createElement('h3');
      h3.textContent = 'Not configured';
      const p = document.createElement('p');
      p.textContent = 'Please set up your API credentials first.';
      emptyDiv.appendChild(h3);
      emptyDiv.appendChild(p);
      aliasesList.appendChild(emptyDiv);
      return;
    }

    // Get aliases from API and history
    const [apiAliases, history] = await Promise.all([
      sendMessage({ action: 'getAllForwarders' }),
      sendMessage({ action: 'getHistory' })
    ]);

    // Merge API aliases with history (history has more metadata)
    const aliasMap = new Map();

    // First, add all history entries
    for (const entry of history) {
      const key = `${entry.alias}@${entry.domain}`;
      aliasMap.set(key, { ...entry, fromHistory: true });
    }

    // Then, update with API data (marks as active if exists on server)
    for (const alias of apiAliases || []) {
      const aliasName = alias.alias || alias.name;
      const key = `${aliasName}@${alias.domain}`;
      const existing = aliasMap.get(key);
      if (existing) {
        aliasMap.set(key, { ...existing, active: true });
      } else {
        aliasMap.set(key, {
          alias: aliasName,
          domain: alias.domain,
          destination: alias.destination,
          active: true,
          fromHistory: false
        });
      }
    }

    cachedAliases = Array.from(aliasMap.values());
    cachedAliases.sort((a, b) => {
      // Sort by creation date if available, newest first
      if (a.createdAt && b.createdAt) {
        return new Date(b.createdAt) - new Date(a.createdAt);
      }
      return 0;
    });

    filterAliases();
  } catch (error) {
    aliasesList.textContent = '';
    const emptyDiv = document.createElement('div');
    emptyDiv.className = 'empty-state';
    const h3 = document.createElement('h3');
    h3.textContent = 'Error';
    const p = document.createElement('p');
    p.textContent = error.message;
    emptyDiv.appendChild(h3);
    emptyDiv.appendChild(p);
    aliasesList.appendChild(emptyDiv);
  }
}

/**
 * Filter and display aliases
 */
function filterAliases() {
  const searchTerm = aliasSearch.value.toLowerCase();
  const filter = aliasFilter.value;

  let filtered = cachedAliases;

  // Apply search filter
  if (searchTerm) {
    filtered = filtered.filter(a =>
      a.alias.toLowerCase().includes(searchTerm) ||
      a.domain.toLowerCase().includes(searchTerm) ||
      (a.site && a.site.toLowerCase().includes(searchTerm)) ||
      (a.destination && a.destination.toLowerCase().includes(searchTerm))
    );
  }

  // Apply status filter
  if (filter === 'active') {
    filtered = filtered.filter(a => a.active);
  } else if (filter === 'deleted') {
    filtered = filtered.filter(a => !a.active);
  }

  renderAliases(filtered);
}

/**
 * Create an alias item element
 */
function createAliasItem(alias) {
  const email = `${alias.alias}@${alias.domain}`;
  const isActive = alias.active;

  const item = document.createElement('div');
  item.className = `alias-item ${isActive ? '' : 'inactive'}`;
  item.dataset.alias = alias.alias;
  item.dataset.domain = alias.domain;

  const infoDiv = document.createElement('div');
  infoDiv.className = 'alias-info';

  const emailSpan = document.createElement('span');
  emailSpan.className = 'alias-email';
  emailSpan.textContent = email;
  infoDiv.appendChild(emailSpan);

  const metaDiv = document.createElement('div');
  metaDiv.className = 'alias-meta';

  if (alias.destination) {
    const destSpan = document.createElement('span');
    destSpan.textContent = `→ ${alias.destination}`;
    metaDiv.appendChild(destSpan);
  }

  if (alias.site) {
    const siteSpan = document.createElement('span');
    siteSpan.className = 'alias-site';
    siteSpan.textContent = alias.site;
    metaDiv.appendChild(siteSpan);
  }

  if (alias.createdAt) {
    const dateSpan = document.createElement('span');
    dateSpan.textContent = formatDate(alias.createdAt);
    metaDiv.appendChild(dateSpan);
  }

  if (!isActive) {
    const deletedSpan = document.createElement('span');
    deletedSpan.style.color = 'var(--error)';
    deletedSpan.textContent = 'Deleted';
    metaDiv.appendChild(deletedSpan);
  }

  infoDiv.appendChild(metaDiv);
  item.appendChild(infoDiv);

  const actionsDiv = document.createElement('div');
  actionsDiv.className = 'alias-actions';

  const copyBtn = document.createElement('button');
  copyBtn.className = 'btn-icon copy';
  copyBtn.title = 'Copy to clipboard';
  copyBtn.textContent = '\u{1F4CB}';
  copyBtn.addEventListener('click', () => {
    copyToClipboard(email);
    showToast('Copied to clipboard');
  });
  actionsDiv.appendChild(copyBtn);

  if (isActive) {
    const deleteBtn = document.createElement('button');
    deleteBtn.className = 'btn-icon delete';
    deleteBtn.title = 'Delete alias';
    deleteBtn.textContent = '\u{1F5D1}';
    deleteBtn.addEventListener('click', () => handleDeleteAlias(alias.alias, alias.domain));
    actionsDiv.appendChild(deleteBtn);
  } else {
    const recreateBtn = document.createElement('button');
    recreateBtn.className = 'btn-icon recreate';
    recreateBtn.title = 'Recreate alias';
    recreateBtn.textContent = '\u267B';
    recreateBtn.addEventListener('click', () => handleRecreateAlias(alias.alias, alias.domain));
    actionsDiv.appendChild(recreateBtn);
  }

  item.appendChild(actionsDiv);
  return item;
}

/**
 * Render aliases list
 */
function renderAliases(aliases) {
  aliasesList.textContent = '';

  if (aliases.length === 0) {
    const emptyDiv = document.createElement('div');
    emptyDiv.className = 'empty-state';
    const iconDiv = document.createElement('div');
    iconDiv.className = 'empty-state-icon';
    iconDiv.textContent = '\u{1F4E7}';
    emptyDiv.appendChild(iconDiv);
    const h3 = document.createElement('h3');
    h3.textContent = 'No aliases found';
    emptyDiv.appendChild(h3);
    const p = document.createElement('p');
    p.textContent = 'Create your first alias using the extension popup.';
    emptyDiv.appendChild(p);
    aliasesList.appendChild(emptyDiv);
    return;
  }

  for (const alias of aliases) {
    aliasesList.appendChild(createAliasItem(alias));
  }
}

/**
 * Handle deleting an alias
 */
async function handleDeleteAlias(alias, domain) {
  if (!confirm(`Delete alias ${alias}@${domain}?`)) {
    return;
  }

  try {
    await sendMessage({ action: 'deleteAlias', alias, domain });
    showToast('Alias deleted');
    await loadAliases();
  } catch (error) {
    showToast(`Error: ${error.message}`, 'error');
  }
}

/**
 * Handle recreating an alias
 */
async function handleRecreateAlias(alias, domain) {
  try {
    // Get the original destination from history
    const history = await sendMessage({ action: 'getHistory' });
    const entry = history.find(h => h.alias === alias && h.domain === domain);

    if (!entry || !entry.destination) {
      showToast('Cannot recreate: missing destination', 'error');
      return;
    }

    await sendMessage({ action: 'recreateAlias', alias, domain, destination: entry.destination });
    showToast('Alias recreated');
    await loadAliases();
  } catch (error) {
    showToast(`Error: ${error.message}`, 'error');
  }
}

/**
 * Send message to background script
 */
function sendMessage(message) {
  return new Promise((resolve, reject) => {
    chrome.runtime.sendMessage(message, response => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else if (response && response.error) {
        reject(new Error(response.error));
      } else {
        resolve(response);
      }
    });
  });
}

/**
 * Show status message
 */
function showStatus(element, message, type) {
  element.textContent = message;
  element.className = `status-message ${type}`;
  element.classList.remove('hidden');

  setTimeout(() => {
    element.classList.add('hidden');
  }, 5000);
}

/**
 * Show toast notification
 */
function showToast(message, type = 'success') {
  const toast = document.createElement('div');
  toast.className = `toast ${type}`;
  toast.textContent = message;

  document.getElementById('toast-container').appendChild(toast);

  setTimeout(() => {
    toast.remove();
  }, 3000);
}

/**
 * Copy text to clipboard
 */
async function copyToClipboard(text) {
  try {
    await navigator.clipboard.writeText(text);
  } catch {
    // Fallback
    const textarea = document.createElement('textarea');
    textarea.value = text;
    document.body.appendChild(textarea);
    textarea.select();
    document.execCommand('copy');
    document.body.removeChild(textarea);
  }
}

/**
 * Format date
 */
function formatDate(dateStr) {
  const date = new Date(dateStr);
  return date.toLocaleDateString(undefined, {
    year: 'numeric',
    month: 'short',
    day: 'numeric'
  });
}

/**
 * Debounce function
 */
function debounce(fn, delay) {
  let timeout;
  return (...args) => {
    clearTimeout(timeout);
    timeout = setTimeout(() => fn(...args), delay);
  };
}

// Initialize
document.addEventListener('DOMContentLoaded', init);
