/**
 * MXroute API wrapper for managing email aliases
 * API docs: https://api.mxroute.com/docs
 */

const API_BASE = 'https://api.mxroute.com';

export class MXrouteAPI {
  constructor(credentials) {
    this.server = credentials.server;
    this.username = credentials.username;
    this.apiKey = credentials.apiKey;
  }

  /**
   * Get authentication headers for API requests
   */
  getHeaders() {
    return {
      'Content-Type': 'application/json',
      'X-Server': this.server,
      'X-Username': this.username,
      'X-API-Key': this.apiKey
    };
  }

  /**
   * Make an authenticated API request
   */
  async request(endpoint, options = {}) {
    const url = `${API_BASE}${endpoint}`;

    let response;
    try {
      response = await fetch(url, {
        ...options,
        headers: {
          ...this.getHeaders(),
          ...options.headers
        }
      });
    } catch (fetchError) {
      const errMsg = fetchError?.message || fetchError?.name || 'Unknown network error';
      throw new Error(`Network error: ${errMsg}`);
    }

    // Handle no-content responses (204)
    if (response.status === 204) {
      return { success: true };
    }

    const data = await response.json();

    if (!response.ok) {
      if (response.status === 401) {
        throw new Error('Authentication failed. Check server, username, and API key.');
      }
      const errorMsg = data?.error?.message || data?.message || `API error: ${response.status}`;
      throw new Error(errorMsg);
    }

    return data;
  }

  /**
   * List all domains for the account
   * @returns {Promise<Array>} List of domain names
   */
  async listDomains() {
    const response = await this.request('/domains');
    return response.data || [];
  }

  /**
   * List all forwarders (aliases) for a domain
   * @param {string} domain - The domain name
   * @returns {Promise<Array>} List of forwarder objects
   */
  async listForwarders(domain) {
    const response = await this.request(`/domains/${encodeURIComponent(domain)}/forwarders`);
    return response.data || [];
  }

  /**
   * Create a new forwarder (alias)
   * @param {string} domain - The domain name
   * @param {string} alias - The alias prefix (part before @)
   * @param {string} destination - The destination email address
   * @returns {Promise<Object>} The created forwarder
   */
  async createForwarder(domain, alias, destination) {
    const response = await this.request(`/domains/${encodeURIComponent(domain)}/forwarders`, {
      method: 'POST',
      body: JSON.stringify({
        alias: alias,
        destinations: [destination]
      })
    });
    return response.data || response;
  }

  /**
   * Delete a forwarder (alias)
   * @param {string} domain - The domain name
   * @param {string} alias - The alias prefix (part before @)
   * @returns {Promise<void>}
   */
  async deleteForwarder(domain, alias) {
    await this.request(`/domains/${encodeURIComponent(domain)}/forwarders/${encodeURIComponent(alias)}`, {
      method: 'DELETE'
    });
    return { success: true };
  }

  /**
   * Test the API connection with current credentials
   * @returns {Promise<boolean>} True if connection is successful
   */
  async testConnection() {
    await this.listDomains();
    return true;
  }
}

/**
 * Generate an alias based on the specified format
 * @param {string} format - 'domain', 'random', or 'manual'
 * @param {string} siteDomain - The current site's domain
 * @returns {string} The generated alias prefix
 */
export function generateAlias(format, siteDomain) {
  switch (format) {
    case 'domain':
      // Extract main domain name (e.g., 'amazon' from 'www.amazon.com')
      const parts = siteDomain.replace(/^www\./, '').split('.');
      const mainPart = parts[0];
      // Sanitize: only alphanumeric and hyphens
      return mainPart.toLowerCase().replace(/[^a-z0-9-]/g, '');

    case 'random':
      // Generate a random string
      const chars = 'abcdefghijklmnopqrstuvwxyz0123456789';
      let result = '';
      for (let i = 0; i < 8; i++) {
        result += chars.charAt(Math.floor(Math.random() * chars.length));
      }
      return result;

    case 'manual':
    default:
      return '';
  }
}

/**
 * Sanitize an alias to only contain valid characters
 * @param {string} alias - The raw alias input
 * @returns {string} The sanitized alias
 */
export function sanitizeAlias(alias) {
  return alias.toLowerCase()
    .replace(/[^a-z0-9._-]/g, '')
    .replace(/^[.-]+/, '')  // Can't start with . or -
    .substring(0, 64);  // Max length
}

/**
 * Validate an email address format
 * @param {string} email - The email address to validate
 * @returns {boolean} True if valid
 */
export function isValidEmail(email) {
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  return emailRegex.test(email);
}
