/**
 * MXroute Alias Manager - Content Script
 * Detects email fields and provides alias creation prompts
 */

// Track processed fields and prompts
const processedFields = new WeakSet();
const fieldPrompts = new WeakMap();

// Settings cache
let settings = null;
let isConfigured = false;

/**
 * Initialize content script
 */
async function init() {
  // Check if extension is configured
  try {
    const response = await sendMessage({ action: 'isConfigured' });
    isConfigured = response && response.configured;

    if (!isConfigured) {
      return;
    }

    // Load settings
    settings = await sendMessage({ action: 'getSettings' });

    if (!settings || settings.autoDetect === false) {
      return;
    }

    // Process existing email fields
    processEmailFields();

    // Observe for dynamically added fields
    observeDOM();
  } catch (error) {
    console.error('MXroute: Error initializing content script:', error);
  }
}

/**
 * Process all email input fields on the page
 */
function processEmailFields() {
  const emailFields = findEmailFields();
  emailFields.forEach(processField);
}

/**
 * Find all email input fields on the page
 */
function findEmailFields() {
  const selectors = [
    'input[type="email"]',
    'input[name*="email" i]',
    'input[id*="email" i]',
    'input[placeholder*="email" i]',
    'input[autocomplete="email"]',
    'input[autocomplete="username"]'
  ];

  const fields = document.querySelectorAll(selectors.join(', '));

  // Filter out hidden or disabled fields
  return Array.from(fields).filter(field => {
    const style = window.getComputedStyle(field);
    return style.display !== 'none' &&
           style.visibility !== 'hidden' &&
           !field.disabled &&
           field.offsetParent !== null;
  });
}

/**
 * Process a single email field
 */
function processField(field) {
  if (processedFields.has(field)) {
    return;
  }

  processedFields.add(field);

  // Create prompt element
  const prompt = createPrompt(field);
  if (prompt) {
    fieldPrompts.set(field, prompt);
    positionPrompt(field, prompt);

    // Reposition on scroll and resize
    const repositionHandler = () => positionPrompt(field, prompt);
    window.addEventListener('scroll', repositionHandler, { passive: true });
    window.addEventListener('resize', repositionHandler, { passive: true });

    // Show/hide based on field focus and value
    field.addEventListener('focus', () => {
      if (!field.value) {
        showPrompt(prompt);
      }
    });

    field.addEventListener('blur', () => {
      // Delay to allow click on prompt
      setTimeout(() => {
        if (!prompt.contains(document.activeElement)) {
          hidePrompt(prompt);
        }
      }, 200);
    });

    field.addEventListener('input', () => {
      if (field.value) {
        hidePrompt(prompt);
      }
    });

    // Show if field is focused and empty
    if (document.activeElement === field && !field.value) {
      showPrompt(prompt);
    }
  }
}

/**
 * Create the alias prompt element
 */
function createPrompt(field) {
  const prompt = document.createElement('div');
  prompt.className = 'mxroute-alias-prompt';
  prompt.style.display = 'none';

  // Icon
  const icon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
  icon.setAttribute('class', 'mxroute-alias-prompt-icon');
  icon.setAttribute('viewBox', '0 0 24 24');
  icon.setAttribute('width', '14');
  icon.setAttribute('height', '14');

  const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
  path.setAttribute('d', 'M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z');
  path.setAttribute('fill', 'none');
  path.setAttribute('stroke', 'currentColor');
  path.setAttribute('stroke-width', '2');

  const path2 = document.createElementNS('http://www.w3.org/2000/svg', 'polyline');
  path2.setAttribute('points', '22,6 12,13 2,6');
  path2.setAttribute('fill', 'none');
  path2.setAttribute('stroke', 'currentColor');
  path2.setAttribute('stroke-width', '2');

  icon.appendChild(path);
  icon.appendChild(path2);
  prompt.appendChild(icon);

  // Text
  const text = document.createElement('span');
  text.className = 'mxroute-alias-prompt-text';
  text.textContent = 'Create alias';
  prompt.appendChild(text);

  // Click handler
  prompt.addEventListener('click', async (e) => {
    e.preventDefault();
    e.stopPropagation();

    if (settings.aliasFormat === 'manual') {
      showManualModal(field);
    } else {
      await createAliasForField(field, prompt);
    }
  });

  document.body.appendChild(prompt);
  return prompt;
}

/**
 * Position the prompt relative to the field
 */
function positionPrompt(field, prompt) {
  const rect = field.getBoundingClientRect();
  const scrollX = window.scrollX || document.documentElement.scrollLeft;
  const scrollY = window.scrollY || document.documentElement.scrollTop;

  // Position at the right edge of the field
  prompt.style.top = `${rect.top + scrollY + (rect.height - prompt.offsetHeight) / 2}px`;
  prompt.style.left = `${rect.right + scrollX - prompt.offsetWidth - 8}px`;
}

/**
 * Show prompt
 */
function showPrompt(prompt) {
  prompt.style.display = 'flex';
}

/**
 * Hide prompt
 */
function hidePrompt(prompt) {
  prompt.style.display = 'none';
}

/**
 * Create alias automatically and fill field
 */
async function createAliasForField(field, prompt) {
  prompt.classList.add('creating');
  const textEl = prompt.querySelector('.mxroute-alias-prompt-text');
  const originalText = textEl.textContent;
  textEl.textContent = 'Creating...';

  try {
    // Get settings and generate alias
    const settings = await sendMessage({ action: 'getSettings' });
    const domains = await sendMessage({ action: 'getDomains' });
    const destinations = await sendMessage({ action: 'getDomainDestinations' });

    // Get domain with destination
    let domain = settings.defaultDomain;
    if (!domain || !destinations[domain]) {
      // Find first domain with destination
      for (const d of domains) {
        const name = typeof d === 'string' ? d : d.domain;
        if (destinations[name]) {
          domain = name;
          break;
        }
      }
    }

    if (!domain || !destinations[domain]) {
      showToast('Please configure a domain destination in settings', 'error');
      return;
    }

    // Generate alias
    const site = window.location.hostname.replace(/^www\./, '');
    const alias = await sendMessage({ action: 'generateAlias', site });

    // Create alias
    const result = await sendMessage({
      action: 'createAlias',
      domain: domain,
      alias: alias,
      destination: destinations[domain],
      site: site
    });

    const fullAlias = result.fullAlias;

    // Fill the field
    fillField(field, fullAlias);

    // Copy to clipboard
    await navigator.clipboard.writeText(fullAlias).catch(() => {});

    // Show success
    showToast(`Created: ${fullAlias}`);

    // Hide prompt
    hidePrompt(prompt);

  } catch (error) {
    showToast(`Error: ${error.message}`, 'error');
  } finally {
    prompt.classList.remove('creating');
    textEl.textContent = originalText;
  }
}

/**
 * Show manual alias entry modal
 */
function showManualModal(field) {
  // Remove existing modal
  const existing = document.querySelector('.mxroute-modal-overlay');
  if (existing) {
    existing.remove();
  }

  // Create modal
  const overlay = document.createElement('div');
  overlay.className = 'mxroute-modal-overlay';

  const modal = document.createElement('div');
  modal.className = 'mxroute-modal';

  const title = document.createElement('div');
  title.className = 'mxroute-modal-title';
  title.textContent = 'Create Email Alias';
  modal.appendChild(title);

  const input = document.createElement('input');
  input.className = 'mxroute-modal-input';
  input.type = 'text';
  input.placeholder = 'Enter alias name...';
  modal.appendChild(input);

  const select = document.createElement('select');
  select.className = 'mxroute-modal-select';
  modal.appendChild(select);

  const buttons = document.createElement('div');
  buttons.className = 'mxroute-modal-buttons';

  const cancelBtn = document.createElement('button');
  cancelBtn.className = 'mxroute-modal-btn mxroute-modal-btn-secondary';
  cancelBtn.textContent = 'Cancel';
  cancelBtn.addEventListener('click', () => overlay.remove());
  buttons.appendChild(cancelBtn);

  const createBtn = document.createElement('button');
  createBtn.className = 'mxroute-modal-btn mxroute-modal-btn-primary';
  createBtn.textContent = 'Create';
  buttons.appendChild(createBtn);

  modal.appendChild(buttons);
  overlay.appendChild(modal);
  document.body.appendChild(overlay);

  // Load domains
  loadDomainsForModal(select);

  // Focus input
  setTimeout(() => input.focus(), 100);

  // Handle create
  createBtn.addEventListener('click', async () => {
    const alias = input.value.trim();
    const domain = select.value;

    if (!alias) {
      input.focus();
      return;
    }

    const destinations = await sendMessage({ action: 'getDomainDestinations' });
    const destination = destinations[domain];

    if (!destination) {
      showToast('No destination configured for this domain', 'error');
      return;
    }

    try {
      const result = await sendMessage({
        action: 'createAlias',
        domain: domain,
        alias: alias,
        destination: destination,
        site: window.location.hostname.replace(/^www\./, '')
      });

      const fullAlias = result.fullAlias;
      fillField(field, fullAlias);
      await navigator.clipboard.writeText(fullAlias).catch(() => {});
      showToast(`Created: ${fullAlias}`);
      overlay.remove();

      // Hide prompt for this field
      const prompt = fieldPrompts.get(field);
      if (prompt) {
        hidePrompt(prompt);
      }
    } catch (error) {
      showToast(`Error: ${error.message}`, 'error');
    }
  });

  // Enter to create
  input.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') {
      createBtn.click();
    } else if (e.key === 'Escape') {
      overlay.remove();
    }
  });

  // Click outside to close
  overlay.addEventListener('click', (e) => {
    if (e.target === overlay) {
      overlay.remove();
    }
  });
}

/**
 * Load domains for modal select
 */
async function loadDomainsForModal(select) {
  try {
    const domains = await sendMessage({ action: 'getDomains' });
    const settings = await sendMessage({ action: 'getSettings' });
    const destinations = await sendMessage({ action: 'getDomainDestinations' });

    select.textContent = '';

    const domainsWithDest = domains.filter(d => {
      const name = typeof d === 'string' ? d : d.domain;
      return destinations[name];
    });

    const availableDomains = domainsWithDest.length > 0 ? domainsWithDest : domains;

    for (const domain of availableDomains) {
      const name = typeof domain === 'string' ? domain : domain.domain;
      const opt = document.createElement('option');
      opt.value = name;
      opt.textContent = name;
      select.appendChild(opt);
    }

    if (settings.defaultDomain) {
      select.value = settings.defaultDomain;
    }
  } catch (error) {
    const opt = document.createElement('option');
    opt.value = '';
    opt.textContent = 'Error loading domains';
    select.appendChild(opt);
  }
}

/**
 * Fill field with email
 */
function fillField(field, email) {
  field.value = email;
  field.dispatchEvent(new Event('input', { bubbles: true }));
  field.dispatchEvent(new Event('change', { bubbles: true }));
}

/**
 * Show toast notification
 */
function showToast(message, type = 'success') {
  // Remove existing toast
  const existing = document.querySelector('.mxroute-toast');
  if (existing) {
    existing.remove();
  }

  const toast = document.createElement('div');
  toast.className = `mxroute-toast ${type}`;
  toast.textContent = message;
  document.body.appendChild(toast);

  setTimeout(() => {
    toast.remove();
  }, 3000);
}

/**
 * Observe DOM for dynamically added email fields
 */
function observeDOM() {
  const observer = new MutationObserver((mutations) => {
    let shouldProcess = false;

    for (const mutation of mutations) {
      if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
        shouldProcess = true;
        break;
      }
    }

    if (shouldProcess) {
      // Debounce processing
      clearTimeout(observeDOM.timeout);
      observeDOM.timeout = setTimeout(processEmailFields, 500);
    }
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

/**
 * Send message to background script
 */
function sendMessage(message) {
  return new Promise((resolve, reject) => {
    chrome.runtime.sendMessage(message, response => {
      if (chrome.runtime.lastError) {
        reject(new Error(chrome.runtime.lastError.message));
      } else if (response && response.error) {
        reject(new Error(response.error));
      } else {
        resolve(response);
      }
    });
  });
}

/**
 * Handle messages from popup/background
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === 'fillEmail') {
    // Find the active/focused email field
    const activeElement = document.activeElement;
    if (activeElement && activeElement.tagName === 'INPUT') {
      fillField(activeElement, message.email);
      sendResponse({ success: true });
    } else {
      // Find first empty email field
      const emailFields = findEmailFields();
      const emptyField = emailFields.find(f => !f.value);
      if (emptyField) {
        fillField(emptyField, message.email);
        sendResponse({ success: true });
      } else {
        sendResponse({ success: false, error: 'No empty email field found' });
      }
    }
  }
  return true;
});

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
