/**
 * MXroute Alias Manager - Service Worker
 * Handles API calls, storage management, and message passing
 */

import { MXrouteAPI, generateAlias, sanitizeAlias } from '../lib/api.js';

// Cache for API instance
let apiInstance = null;

/**
 * Get or create API instance with current credentials
 */
async function getAPI() {
  const { credentials } = await chrome.storage.sync.get('credentials');
  if (!credentials || !credentials.server || !credentials.username || !credentials.apiKey) {
    throw new Error('API credentials not configured');
  }

  // Check if credentials changed
  if (apiInstance &&
      apiInstance.server === credentials.server &&
      apiInstance.username === credentials.username &&
      apiInstance.apiKey === credentials.apiKey) {
    return apiInstance;
  }

  apiInstance = new MXrouteAPI(credentials);
  return apiInstance;
}

/**
 * Get settings with defaults
 */
async function getSettings() {
  const { settings } = await chrome.storage.sync.get('settings');
  return {
    aliasFormat: 'domain',
    autoDetect: true,
    defaultDomain: null,
    ...settings
  };
}

/**
 * Get domain destinations
 */
async function getDomainDestinations() {
  const { domainDestinations } = await chrome.storage.sync.get('domainDestinations');
  return domainDestinations || {};
}

/**
 * Get alias history
 */
async function getAliasHistory() {
  const { aliasHistory } = await chrome.storage.sync.get('aliasHistory');
  return aliasHistory || [];
}

/**
 * Save alias to history
 */
async function saveToHistory(aliasData) {
  const history = await getAliasHistory();

  // Check if alias already exists in history
  const existingIndex = history.findIndex(
    h => h.alias === aliasData.alias && h.domain === aliasData.domain
  );

  if (existingIndex >= 0) {
    // Update existing entry
    history[existingIndex] = { ...history[existingIndex], ...aliasData, active: true };
  } else {
    // Add new entry at the beginning
    history.unshift(aliasData);
  }

  // Keep only last 500 entries
  const trimmedHistory = history.slice(0, 500);

  await chrome.storage.sync.set({ aliasHistory: trimmedHistory });
  return trimmedHistory;
}

/**
 * Mark alias as inactive in history (soft delete)
 */
async function markAliasInactive(alias, domain) {
  const history = await getAliasHistory();
  const index = history.findIndex(
    h => h.alias === alias && h.domain === domain
  );

  if (index >= 0) {
    history[index].active = false;
    history[index].deletedAt = new Date().toISOString();
    await chrome.storage.sync.set({ aliasHistory: history });
  }

  return history;
}

/**
 * Handle messages from popup and content scripts
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  handleMessage(message, sender)
    .then(sendResponse)
    .catch(error => sendResponse({ error: error.message }));
  return true; // Keep channel open for async response
});

/**
 * Process incoming messages
 */
async function handleMessage(message, sender) {
  const { action } = message;

  switch (action) {
    case 'testConnection':
      return testConnection(message.credentials);

    case 'getDomains':
      return getDomains();

    case 'getForwarders':
      return getForwarders(message.domain);

    case 'getAllForwarders':
      return getAllForwarders();

    case 'createAlias':
      return createAlias(message.domain, message.alias, message.destination, message.site);

    case 'deleteAlias':
      return deleteAlias(message.domain, message.alias);

    case 'recreateAlias':
      return recreateAlias(message.alias, message.domain, message.destination);

    case 'generateAlias':
      return generateAliasForSite(message.site);

    case 'getSettings':
      return getSettings();

    case 'saveSettings':
      await chrome.storage.sync.set({ settings: message.settings });
      return { success: true };

    case 'getCredentials':
      const { credentials } = await chrome.storage.sync.get('credentials');
      return credentials || null;

    case 'saveCredentials':
      await chrome.storage.sync.set({ credentials: message.credentials });
      apiInstance = null; // Reset API instance
      return { success: true };

    case 'getDomainDestinations':
      return getDomainDestinations();

    case 'saveDomainDestinations':
      await chrome.storage.sync.set({ domainDestinations: message.destinations });
      return { success: true };

    case 'getHistory':
      return getAliasHistory();

    case 'searchHistory':
      return searchHistory(message.query);

    case 'getRecentAliases':
      return getRecentAliases(message.limit || 5);

    case 'isConfigured':
      return isConfigured();

    default:
      throw new Error(`Unknown action: ${action}`);
  }
}

/**
 * Test API connection with given credentials
 */
async function testConnection(credentials) {
  const api = new MXrouteAPI(credentials);
  try {
    const domains = await api.listDomains();
    return { success: true, domains };
  } catch (error) {
    return { success: false, error: error.message || String(error) };
  }
}

/**
 * Get all domains
 */
async function getDomains() {
  const api = await getAPI();
  const domains = await api.listDomains();

  // Cache domains
  await chrome.storage.local.set({ cachedDomains: domains });

  return domains;
}

/**
 * Get forwarders for a specific domain
 */
async function getForwarders(domain) {
  const api = await getAPI();
  return api.listForwarders(domain);
}

/**
 * Get all forwarders across all domains
 */
async function getAllForwarders() {
  const domains = await getDomains();
  const allForwarders = [];

  for (const domain of domains) {
    try {
      const domainName = typeof domain === 'string' ? domain : domain.domain;
      const forwarders = await getForwarders(domainName);

      // Add domain info to each forwarder
      const forwardersArray = Array.isArray(forwarders) ? forwarders : [];
      for (const fwd of forwardersArray) {
        allForwarders.push({
          ...fwd,
          domain: domainName
        });
      }
    } catch (error) {
      console.error(`Error fetching forwarders for ${domain}:`, error);
    }
  }

  return allForwarders;
}

/**
 * Create a new alias
 */
async function createAlias(domain, alias, destination, site) {
  const api = await getAPI();
  const sanitized = sanitizeAlias(alias);

  if (!sanitized) {
    throw new Error('Invalid alias format');
  }

  const result = await api.createForwarder(domain, sanitized, destination);

  // Save to history
  const historyEntry = {
    alias: sanitized,
    domain: domain,
    destination: destination,
    createdAt: new Date().toISOString(),
    site: site || null,
    active: true
  };

  await saveToHistory(historyEntry);

  return {
    ...result,
    fullAlias: `${sanitized}@${domain}`,
    historyEntry
  };
}

/**
 * Delete an alias
 */
async function deleteAlias(domain, alias) {
  const api = await getAPI();
  await api.deleteForwarder(domain, alias);

  // Mark as inactive in history
  await markAliasInactive(alias, domain);

  return { success: true };
}

/**
 * Recreate a previously deleted alias
 */
async function recreateAlias(alias, domain, destination) {
  const api = await getAPI();
  const result = await api.createForwarder(domain, alias, destination);

  // Update history entry
  const history = await getAliasHistory();
  const index = history.findIndex(
    h => h.alias === alias && h.domain === domain
  );

  if (index >= 0) {
    history[index].active = true;
    history[index].recreatedAt = new Date().toISOString();
    delete history[index].deletedAt;
    await chrome.storage.sync.set({ aliasHistory: history });
  }

  return {
    ...result,
    fullAlias: `${alias}@${domain}`
  };
}

/**
 * Generate alias for a specific site
 */
async function generateAliasForSite(site) {
  const settings = await getSettings();
  return generateAlias(settings.aliasFormat, site);
}

/**
 * Search alias history
 */
async function searchHistory(query) {
  const history = await getAliasHistory();
  const lowerQuery = query.toLowerCase();

  return history.filter(entry =>
    entry.alias.toLowerCase().includes(lowerQuery) ||
    entry.domain.toLowerCase().includes(lowerQuery) ||
    (entry.site && entry.site.toLowerCase().includes(lowerQuery)) ||
    entry.destination.toLowerCase().includes(lowerQuery)
  );
}

/**
 * Get recent active aliases
 */
async function getRecentAliases(limit = 5) {
  const history = await getAliasHistory();
  return history.filter(h => h.active).slice(0, limit);
}

/**
 * Check if extension is configured
 */
async function isConfigured() {
  const { credentials } = await chrome.storage.sync.get('credentials');
  const configured = !!(credentials && credentials.server && credentials.username && credentials.apiKey);
  return { configured };
}

// Handle installation
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    // Open options page on first install
    chrome.runtime.openOptionsPage();
  }
});
